const express = require('express');
const { Client, LocalAuth } = require('whatsapp-web.js');
const qrcode = require('qrcode');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const path = require('path');
const dotenv = require('dotenv');

dotenv.config();
const app = express();
const server = http.createServer(app);
const io = socketIo(server);

app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, 'public')));

const client = new Client({
  authStrategy: new LocalAuth()
});

let isReady = false;

client.on('qr', (qr) => {
  qrcode.toDataURL(qr, (err, url) => {
    io.emit('qr', url);
    io.emit('message', 'QR generado, escanéalo con WhatsApp');
    isReady = false;
  });
});

client.on('ready', () => {
  isReady = true;
  io.emit('ready', 'WhatsApp autenticado');
  io.emit('message', 'WhatsApp está listo para enviar mensajes');
});

client.on('authenticated', () => {
  io.emit('message', 'Autenticado con éxito');
});

client.on('auth_failure', () => {
  isReady = false;
  io.emit('message', 'Error de autenticación, reinicia el servidor');
});

client.on('disconnected', () => {
  isReady = false;
  io.emit('message', 'Cliente desconectado');
});

client.initialize();

// API para enviar mensajes
app.post('/api/send', async (req, res) => {
  if (!isReady) return res.status(400).json({ error: 'WhatsApp no autenticado' });

  const { number, message } = req.body;
  if (!number || !message) return res.status(400).json({ error: 'Faltan datos' });

  const chatId = number.includes('@c.us') ? number : `${number}@c.us`;

  try {
    // Verifica si el número está registrado en WhatsApp
    const isRegistered = await client.isRegisteredUser(chatId);
    if (!isRegistered) {
      return res.status(400).json({ error: 'El número no está registrado en WhatsApp' });
    }

    await client.sendMessage(chatId, message);
    res.json({ status: 'Mensaje enviado' });
    io.emit('message', `Mensaje enviado a ${number}`);
  } catch (err) {
    console.log(err);
    res.status(500).json({ error: err.message });
    io.emit('message', `Error al enviar mensaje: ${err.message}`);
  }
});

// Servir el frontend
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

const PORT = process.env.PORT || 3000;
server.listen(PORT, () => {
  console.log(`Servidor corriendo en http://localhost:${PORT}`);
});